<?php
/*
Plugin Name: Dokan Manual Vendor Approval (v2)
Description: Require manual approval for new Dokan vendors + email notifications. Improved: detects role assignment and prevents auto-approval.
Version: 1.1
Author: Youssef
*/

// Helper: set user to pending vendor
function dma_set_user_pending($user_id) {
    $user = get_userdata($user_id);
    if (!$user) return;

    // Prevent changing if already pending
    wp_update_user([
        'ID' => $user_id,
        'user_status' => 1 // pending
    ]);

    update_user_meta($user_id, 'dokan_enable_selling', 'no'); // disable selling

    // Send email to admin
    wp_mail(
        get_option('admin_email'),
        'New Vendor Registration - Pending Approval',
        "A new vendor has registered.\n\nUsername: {$user->user_login}\nEmail: {$user->user_email}\n\nPlease approve them in WP Admin → Dokan → Vendors."
    );

    // Send email to vendor
    wp_mail(
        $user->user_email,
        'Your Vendor Account is Pending Approval',
        "Thank you for registering as a vendor.\n\nYour account is pending approval. You will receive an email once your account is approved."
    );
}

// Fallback: when WooCommerce creates a customer (Dokan may use this flow)
add_action('woocommerce_created_customer', function($customer_id) {
    $user = get_userdata($customer_id);
    if (!$user) return;

    // If user already has seller role, set pending
    if ( in_array('seller', (array) $user->roles) || in_array('vendor', (array) $user->roles) ) {
        dma_set_user_pending($customer_id);
    }
}, 20, 1);

// Primary: when a plugin sets the user role (Dokan assigns seller role), detect it
add_action('set_user_role', function($user_id, $role, $old_roles) {
    // common Dokan role names
    $vendor_roles = array('seller', 'vendor', 'seller_admin');

    if ( in_array($role, $vendor_roles) ) {
        dma_set_user_pending($user_id);
    }
}, 10, 3);

// If registration creates user with seller role via user_register, handle later (priority)
add_action('user_register', function ($user_id) {
    $user = get_userdata($user_id);
    if (!$user) return;

    if ( in_array('seller', (array) $user->roles) || in_array('vendor', (array) $user->roles) ) {
        dma_set_user_pending($user_id);
    }
}, 20);

// When admin enables vendor in Dokan, approve them
add_action('dokan_vendor_enabled', function ($vendor_id) {

    update_user_meta($vendor_id, 'dokan_enable_selling', 'yes');

    wp_update_user([
        'ID' => $vendor_id,
        'user_status' => 0 // approved
    ]);

    // Send approval email
    $user = get_userdata($vendor_id);
    if ($user) {
        wp_mail(
            $user->user_email,
            'Your Vendor Account Has Been Approved',
            "Good news! Your vendor account has been approved. You may now log in and start selling."
        );
    }
});
