<?php
/**
 * Template Name: Purchase Details
 */

global $wp;

// Get order ID from URL parameter
$order_id = isset( $_GET['order_id'] ) ? intval( $_GET['order_id'] ) : 0;

// Verify order exists and belongs to current user
$order = wc_get_order( $order_id );
$current_user_id = get_current_user_id();

if ( ! $order || $order->get_customer_id() != $current_user_id ) {
    echo '<div class="dokan-dashboard-content dokan-purchase-details-content">';
    echo '<div class="dokan-alert dokan-alert-danger">';
    echo __( 'Commande non trouvée ou vous n\'avez pas la permission de la voir.', 'glowup' );
    echo '</div>';
    echo '<a href="' . esc_url( dokan_get_navigation_url( 'my-purchases' ) ) . '" class="dokan-btn dokan-btn-default">';
    echo __( 'Retour à Mes Achats', 'glowup' );
    echo '</a>';
    echo '</div>';
    return;
}

// Get order details
$order_number = $order->get_order_number();
$order_date = $order->get_date_created();
$order_status = $order->get_status();
$order_total = $order->get_formatted_order_total();
$payment_method = $order->get_payment_method_title();

// Check if cancellation is possible
$can_cancel = false;
$cancellable_statuses = apply_filters( 'woocommerce_valid_order_statuses_for_cancel', array( 'pending', 'failed', 'on-hold', 'processing', 'shipping' ), $order );

if ( in_array( $order->get_status(), $cancellable_statuses ) ) {
    $can_cancel = true;
}

// Handle cancellation request
if ( $can_cancel && isset( $_POST['cancel_order'] ) && isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'dokan_cancel_order_' . $order_id ) ) {
    // Cancel the order
    $order->update_status( 'cancelled', __( 'Commande annulée par le client.', 'woocommerce' ) );
    
    // Show success message - UPDATED TO FIXED POSITION
    echo '<div class="dokan-alert dokan-alert-success dokan-cancellation-success dokan-cancellation-success-fixed">';
    echo '<div class="dokan-cancellation-success-content">';
    echo '<i class="fas fa-check-circle"></i> ' . __( 'Votre commande a été annulée avec succès.', 'glowup' );
    echo '</div>';
    echo '</div>';
    
    // Refresh order data
    $order = wc_get_order( $order_id );
    $order_status = $order->get_status();
    $can_cancel = false;
}
?>

<!-- DOKAN DASHBOARD WRAPPER -->
<div class="dokan-dashboard-wrap">
    
    <!-- DOKAN SIDEBAR MENU -->
    <?php 
    dokan_get_template_part( 'global/dashboard-nav', '', array( 'active_menu' => 'my-purchases' ) ); 
    ?>
    
    <!-- MAIN CONTENT AREA -->
    <div class="dokan-dashboard-content dokan-purchase-details-content">
        
        <!-- BACK BUTTON -->
        <div class="dokan-purchase-back">
            <a href="<?php echo dokan_get_navigation_url( 'my-purchases' ); ?>" class="dokan-btn dokan-btn-default">
                <i class="fas fa-arrow-left"></i> <?php _e( 'Retour à Mes Achats', 'glowup' ); ?>
            </a>
        </div>
        
        <!-- ORDER HEADER -->
        <header class="dokan-purchase-header">
            <div class="dokan-purchase-header-top">
                <h1 class="entry-title">
                    <?php printf( __( 'Commande #%s', 'glowup' ), $order_number ); ?>
                    <span class="dokan-label dokan-label-<?php echo esc_attr( $order_status ); ?>">
                        <?php echo esc_html( wc_get_order_status_name( $order_status ) ); ?>
                    </span>
                </h1>
                
                <?php if ( $can_cancel ) : ?>
                <div class="dokan-purchase-header-actions">
                    <button type="button" class="dokan-btn dokan-btn-danger" id="dokan-show-cancel-form">
                        <i class="fas fa-ban"></i> <?php _e( 'Annuler la commande', 'glowup' ); ?>
                    </button>
                </div>
                <?php endif; ?>
            </div>
            <p class="dokan-purchase-date">
                <i class="far fa-calendar"></i> <?php echo esc_html( wc_format_datetime( $order_date ) ); ?>
            </p>
        </header>
        
        <!-- CANCELLATION FORM (Hidden by default) -->
        <?php if ( $can_cancel ) : ?>
        <div class="dokan-cancellation-form" id="dokan-cancellation-form" style="display: none;">
            <div class="dokan-alert dokan-alert-warning">
                <h4><i class="fas fa-exclamation-triangle"></i> <?php _e( 'Êtes-vous sûr de vouloir annuler cette commande?', 'glowup' ); ?></h4>
                <p><?php _e( 'Une fois annulée, cette action ne peut pas être annulée. Tous les articles de la commande seront marqués comme annulés.', 'glowup' ); ?></p>
            </div>
            
            <form method="post" class="dokan-cancellation-form-content">
                <?php wp_nonce_field( 'dokan_cancel_order_' . $order_id, '_wpnonce' ); ?>
                <input type="hidden" name="cancel_order" value="1">
                
                <div class="dokan-form-group">
                    <label for="cancel_reason">
                        <strong><?php _e( 'Raison de l\'annulation (optionnel)', 'glowup' ); ?></strong>
                    </label>
                    <textarea name="cancel_reason" id="cancel_reason" class="dokan-form-control" rows="3" placeholder="<?php esc_attr_e( 'Pourquoi souhaitez-vous annuler cette commande?', 'glowup' ); ?>"></textarea>
                </div>
                
                <div class="dokan-cancellation-actions">
                    <button type="button" class="dokan-btn dokan-btn-default" id="dokan-hide-cancel-form">
                        <?php _e( 'Non, garder la commande', 'glowup' ); ?>
                    </button>
                    <button type="submit" class="dokan-btn dokan-btn-danger">
                        <i class="fas fa-ban"></i> <?php _e( 'Oui, annuler la commande', 'glowup' ); ?>
                    </button>
                </div>
            </form>
        </div>
        <?php endif; ?>
        
        <!-- ORDER DETAILS SECTIONS -->
        <div class="dokan-purchase-sections">
            
            <!-- ORDER ITEMS -->
            <section class="dokan-purchase-section">
                <h3><?php _e( 'Articles commandés', 'glowup' ); ?></h3>
                <table class="dokan-table dokan-table-striped dokan-purchase-items-table">
                    <thead>
                        <tr>
                            <th><?php _e( 'Produit', 'dokan-lite' ); ?></th>
                            <th><?php _e( 'Quantité', 'dokan-lite' ); ?></th>
                            <th><?php _e( 'Prix', 'dokan-lite' ); ?></th>
                            <th><?php _e( 'Total', 'dokan-lite' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        foreach ( $order->get_items() as $item_id => $item ) : 
                            $product = $item->get_product();
                            $product_name = $product ? $product->get_name() : $item->get_name();
                        ?>
                            <tr>
                                <td class="dokan-purchase-product">
                                    <div class="dokan-purchase-product-name">
                                        <?php echo esc_html( $product_name ); ?>
                                    </div>
                                    <?php if ( $product && $product->get_sku() ) : ?>
                                        <div class="dokan-purchase-product-sku">
                                            <small><?php printf( __( 'SKU: %s', 'glowup' ), $product->get_sku() ); ?></small>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td class="dokan-purchase-quantity">
                                    <?php echo esc_html( $item->get_quantity() ); ?>
                                </td>
                                <td class="dokan-purchase-price">
                                    <?php echo wc_price( $order->get_item_subtotal( $item, false, true ) ); ?>
                                </td>
                                <td class="dokan-purchase-total">
                                    <?php echo wc_price( $order->get_line_total( $item, false, true ) ); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </section>
            
            <!-- ORDER TOTALS -->
            <div class="dokan-row">
                <div class="dokan-col-6">
                    <section class="dokan-purchase-section">
                        <h3><?php _e( 'Totaux de la commande', 'glowup' ); ?></h3>
                        <table class="dokan-table dokan-purchase-totals-table">
                            <tbody>
                                <tr>
                                    <td><?php _e( 'Sous-total', 'woocommerce' ); ?></td>
                                    <td><?php echo wc_price( $order->get_subtotal() ); ?></td>
                                </tr>
                                
                                <?php if ( $order->get_shipping_total() ) : ?>
                                <tr>
                                    <td><?php _e( 'Livraison', 'woocommerce' ); ?></td>
                                    <td><?php echo wc_price( $order->get_shipping_total() ); ?></td>
                                </tr>
                                <?php endif; ?>
                                
                                <?php if ( $order->get_total_tax() ) : ?>
                                <tr>
                                    <td><?php _e( 'Taxe', 'woocommerce' ); ?></td>
                                    <td><?php echo wc_price( $order->get_total_tax() ); ?></td>
                                </tr>
                                <?php endif; ?>
                                
                                <tr class="dokan-purchase-total-row">
                                    <td><strong><?php _e( 'Total', 'woocommerce' ); ?></strong></td>
                                    <td><strong><?php echo $order_total; ?></strong></td>
                                </tr>
                            </tbody>
                        </table>
                    </section>
                </div>
                
                <div class="dokan-col-6">
                    <!-- PAYMENT METHOD -->
                    <section class="dokan-purchase-section">
                        <h3><?php _e( 'Moyen de paiement', 'glowup' ); ?></h3>
                        <p><?php echo $payment_method ? esc_html( $payment_method ) : __( 'Non spécifié', 'glowup' ); ?></p>
                    </section>
                    
                    <!-- BILLING ADDRESS -->
                    <section class="dokan-purchase-section">
                        <h3><?php _e( 'Adresse de facturation', 'glowup' ); ?></h3>
                        <address>
                            <?php echo wp_kses_post( $order->get_formatted_billing_address() ); ?>
                        </address>
                    </section>
                    
                    <!-- SHIPPING ADDRESS -->
                    <?php if ( ! wc_ship_to_billing_address_only() && $order->needs_shipping_address() ) : ?>
                    <section class="dokan-purchase-section">
                        <h3><?php _e( 'Adresse de livraison', 'glowup' ); ?></h3>
                        <address>
                            <?php echo wp_kses_post( $order->get_formatted_shipping_address() ); ?>
                        </address>
                    </section>
                    <?php endif; ?>
                </div>
            </div>
            
        </div> <!-- .dokan-purchase-sections -->
        
    </div> <!-- .dokan-dashboard-content -->
    
</div> <!-- .dokan-dashboard-wrap -->

<style>
/* Purchase Details Styling */
.dokan-purchase-details-content {
    padding: 30px;
}

.dokan-purchase-back {
    margin-bottom: 20px;
}

.dokan-purchase-header {
    margin-bottom: 30px;
    padding-bottom: 20px;
    border-bottom: 1px solid #e5e5e5;
}

.dokan-purchase-header-top {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
}

.dokan-purchase-header h1 {
    margin: 0;
    color: #333;
    font-size: 24px;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 15px;
}

.dokan-purchase-header .dokan-label {
    font-size: 14px;
    padding: 6px 15px;
}

.dokan-purchase-date {
    margin: 0;
    color: #666;
    font-size: 14px;
}

.dokan-purchase-date i {
    margin-right: 5px;
}

.dokan-purchase-header-actions {
    display: flex;
    gap: 10px;
}

/* Cancellation Form */
.dokan-cancellation-form {
    background: #fff;
    border: 1px solid #ffc107;
    border-radius: 4px;
    padding: 25px;
    margin-bottom: 25px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
}

.dokan-alert-warning {
    background: #fff3cd;
    border-color: #ffeaa7;
    color: #856404;
    padding: 15px;
    border-radius: 4px;
    margin-bottom: 20px;
}

.dokan-alert-warning h4 {
    margin: 0 0 10px 0;
    color: #856404;
    font-size: 16px;
}

.dokan-alert-warning i {
    margin-right: 8px;
}

.dokan-cancellation-form-content {
    margin-top: 20px;
}

.dokan-form-group {
    margin-bottom: 20px;
}

.dokan-form-group label {
    display: block;
    margin-bottom: 8px;
    color: #333;
}

.dokan-form-control {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
    resize: vertical;
}

.dokan-cancellation-actions {
    display: flex;
    gap: 10px;
    justify-content: flex-end;
}

/* FIXED SUCCESS MESSAGE WITH ANIMATIONS */
.dokan-cancellation-success-fixed {
    position: fixed !important;
    top: 20px !important;
    left: 50% !important;
    transform: translateX(-50%) !important;
    z-index: 999999 !important;
    max-width: 500px !important;
    width: 90% !important;
    box-shadow: 0 4px 20px rgba(0,0,0,0.2) !important;
    border-radius: 8px !important;
    animation: dokan-slide-down 0.5s ease-out !important;
}

.dokan-cancellation-success-content {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 15px 20px !important;
}

.dokan-cancellation-success-content i {
    font-size: 20px;
}

/* Ensure sidebar doesn't overlap */
.dokan-dashboard-nav {
    z-index: 1000;
    position: relative;
}

.dokan-dashboard-content {
    position: relative;
    z-index: 1;
}

/* Animation for success message */
@keyframes dokan-slide-down {
    from {
        transform: translateX(-50%) translateY(-100%);
        opacity: 0;
    }
    to {
        transform: translateX(-50%) translateY(0);
        opacity: 1;
    }
}

/* Auto-hide success message after 5 seconds */
.dokan-cancellation-success-fixed {
    animation: dokan-slide-down 0.5s ease-out, dokan-fade-out 0.5s ease-out 4.5s forwards !important;
}

@keyframes dokan-fade-out {
    from {
        opacity: 1;
        transform: translateX(-50%) translateY(0);
    }
    to {
        opacity: 0;
        transform: translateX(-50%) translateY(-100%);
    }
}

/* Purchase Sections */
.dokan-purchase-sections {
    margin-top: 30px;
}

.dokan-purchase-section {
    background: #fff;
    border: 1px solid #e5e5e5;
    border-radius: 4px;
    padding: 25px;
    margin-bottom: 25px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
}

.dokan-purchase-section h3 {
    margin: 0 0 20px 0;
    color: #333;
    font-size: 18px;
    font-weight: 600;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

/* Items Table */
.dokan-purchase-items-table {
    width: 100%;
    border-collapse: collapse;
}

.dokan-purchase-items-table th {
    background: #f8f9fa;
    padding: 12px 15px;
    text-align: left;
    font-weight: 600;
    color: #495057;
    border-bottom: 2px solid #e9ecef;
}

.dokan-purchase-items-table td {
    padding: 15px;
    border-bottom: 1px solid #e9ecef;
    vertical-align: top;
}

.dokan-purchase-product-name {
    font-weight: 500;
    margin-bottom: 5px;
}

.dokan-purchase-product-sku {
    color: #666;
    font-size: 13px;
}

/* Totals Table */
.dokan-purchase-totals-table {
    width: 100%;
}

.dokan-purchase-totals-table td {
    padding: 10px 0;
    border-bottom: 1px solid #eee;
}

.dokan-purchase-totals-table tr:last-child td {
    border-bottom: none;
}

.dokan-purchase-total-row {
    background: #f8f9fa;
    border-top: 2px solid #dee2e6 !important;
}

.dokan-purchase-total-row td {
    padding-top: 15px;
    padding-bottom: 15px;
    font-size: 16px;
}

/* Address */
.dokan-purchase-section address {
    font-style: normal;
    line-height: 1.6;
    margin: 0;
}

/* Responsive */
@media (max-width: 768px) {
    .dokan-purchase-details-content {
        padding: 15px;
    }
    
    .dokan-purchase-header-top {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .dokan-purchase-header h1 {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }
    
    .dokan-purchase-section {
        padding: 15px;
    }
    
    .dokan-row {
        flex-direction: column;
    }
    
    .dokan-col-6 {
        width: 100%;
    }
    
    .dokan-purchase-items-table {
        display: block;
        overflow-x: auto;
    }
    
    .dokan-cancellation-actions {
        flex-direction: column;
    }
}

/* Status Labels */
.dokan-label {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 3px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
}

.dokan-label-cancelled {
    background: #f8d7da;
    color: #721c24;
}

.dokan-label-completed {
    background: #d4edda;
    color: #155724;
}

.dokan-label-processing {
    background: #cce5ff;
    color: #004085;
}

.dokan-label-shipping{
    background: #8700c5ff;
    color: #000000ff;
}

.dokan-label-on-hold {
    background: #fff3cd;
    color: #856404;
}

.dokan-label-pending {
    background: #ffeaa7;
    color: #856404;
}

.dokan-label-failed {
    background: #f5c6cb;
    color: #721c24;
}

/* Buttons */
.dokan-btn {
    padding: 10px 20px;
    font-size: 14px;
    border: none;
    border-radius: 4px;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: all 0.2s;
    cursor: pointer;
    font-weight: 500;
}

.dokan-btn-default {
    background: #6c757d;
    color: white;
}

.dokan-btn-default:hover {
    background: #5a6268;
    color: white;
}

.dokan-btn-danger {
    background: #dc3545;
    color: white;
}

.dokan-btn-danger:hover {
    background: #c82333;
    color: white;
}

.dokan-btn i {
    font-size: 14px;
}

.dokan-alert {
    padding: 15px;
    border-radius: 4px;
    margin-bottom: 20px;
}

.dokan-alert-success {
    background: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
}
</style>

<script>
jQuery(document).ready(function($) {
    // Show cancellation form
    $('#dokan-show-cancel-form').on('click', function(e) {
        e.preventDefault();
        $('#dokan-cancellation-form').slideDown(300);
        $(this).hide();
    });
    
    // Hide cancellation form
    $('#dokan-hide-cancel-form').on('click', function(e) {
        e.preventDefault();
        $('#dokan-cancellation-form').slideUp(300);
        $('#dokan-show-cancel-form').show();
    });
    
    // Confirm cancellation
    $('form.dokan-cancellation-form-content').on('submit', function(e) {
        var reason = $('#cancel_reason').val();
        var confirmMessage = reason 
            ? "<?php _e( 'Voulez-vous vraiment annuler cette commande?\nRaison: ', 'glowup' ); ?>" + reason
            : "<?php _e( 'Voulez-vous vraiment annuler cette commande sans spécifier de raison?', 'glowup' ); ?>";
        
        if (!confirm(confirmMessage)) {
            e.preventDefault();
        }
    });
    
    // Auto-remove success message after 5 seconds
    setTimeout(function() {
        $('.dokan-cancellation-success-fixed').fadeOut(500, function() {
            $(this).remove();
        });
    }, 5000);
    
    // Also allow manual closing
    $(document).on('click', '.dokan-cancellation-success-fixed', function() {
        $(this).fadeOut(300, function() {
            $(this).remove();
        });
    });
});
</script>